<?php
session_start();
require_once '../config/db.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Get date range from request
$start_date = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date = $_GET['end_date'] ?? date('Y-m-d');

try {
    // Get customers data with sales information
    $stmt = $pdo->prepare("
        SELECT c.*,
               COUNT(DISTINCT s.id) as total_transactions,
               COALESCE(SUM(s.total_amount), 0) as total_spent,
               COALESCE(SUM(s.amount_paid), 0) as total_paid,
               COALESCE(SUM(s.total_amount - s.amount_paid), 0) as total_credit,
               MAX(s.created_at) as last_purchase_date
        FROM customers c
        LEFT JOIN sales s ON c.id = s.customer_id AND DATE(s.created_at) BETWEEN ? AND ?
        GROUP BY c.id
        ORDER BY total_spent DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $customers = $stmt->fetchAll();

    // Set headers for Excel download
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment;filename="customers_report_' . date('Y-m-d') . '.xls"');
    header('Cache-Control: max-age=0');

    // Output Excel content
    echo "Customers Report\n";
    echo "Period: " . date('M d, Y', strtotime($start_date)) . " to " . date('M d, Y', strtotime($end_date)) . "\n\n";

    // Headers
    echo implode("\t", [
        'Customer ID',
        'Name',
        'Phone',
        'Email',
        'Total Transactions',
        'Total Spent',
        'Total Paid',
        'Total Credit',
        'Last Purchase Date'
    ]) . "\n";

    // Data rows
    foreach ($customers as $customer) {
        echo implode("\t", [
            $customer['id'],
            $customer['name'],
            $customer['phone'],
            $customer['email'],
            $customer['total_transactions'],
            number_format($customer['total_spent'], 2),
            number_format($customer['total_paid'], 2),
            number_format($customer['total_credit'], 2),
            $customer['last_purchase_date'] ? date('M d, Y', strtotime($customer['last_purchase_date'])) : 'Never'
        ]) . "\n";
    }

} catch (PDOException $e) {
    die('Error generating report: ' . $e->getMessage());
} 